<?php
/*--------------------------------------------------------------
   Factory.php 2021-06-28
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview;

use Gambio\Admin\Modules\Language\Model\ValueObjects\LanguageCode;
use Gambio\Admin\Modules\Language\Services\LanguageFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option as OptionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Visualization as VisualizationFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Titles;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\WidgetDatabaseEntries;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\WidgetDefinitions;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Widgets;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\Widget;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDatabaseEntry;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Author;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Height;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\HorizontalPosition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Icon;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Id;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Label;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\UpdateSet;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Title;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Type;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Version;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\VerticalPosition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Visualization;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Width;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;

class Factory
{
    /**
     * @var OptionFactory
     */
    private $optionFactory;
    
    /**
     * @var VisualizationFactory
     */
    private $visualizationFactory;
    
    /**
     * @var LanguageFactory
     */
    private $languageFactory;
    
    
    /**
     * Constructor.
     */
    public function __construct(
        OptionFactory $optionFactory,
        VisualizationFactory $visualizationFactory,
        LanguageFactory $languageFactory
    ) {
        $this->optionFactory        = $optionFactory;
        $this->visualizationFactory = $visualizationFactory;
        $this->languageFactory      = $languageFactory;
    }
    
    
    /**
     * Return option factory.
     */
    public function useOptions(): OptionFactory
    {
        return $this->optionFactory;
    }
    
    
    /**
     * Return visualization factory.
     */
    public function useVisualizations(): VisualizationFactory
    {
        return $this->visualizationFactory;
    }
    
    
    /**
     * Return widget.
     */
    public function createWidget(
        Id $id,
        Label $label,
        Icon $icon,
        Width $width,
        Height $height,
        HorizontalPosition $horizontalPosition,
        VerticalPosition $verticalPosition,
        Visualization $visualization,
        Options $options,
        Data $data
    ): Widget {
        return Widget::create($id,
                              $label,
                              $icon,
                              $width,
                              $height,
                              $horizontalPosition,
                              $verticalPosition,
                              $visualization,
                              $options,
                              $data);
    }
    
    
    /**
     * Return widgets.
     */
    public function createWidgets(Widget ...$widgets): Widgets
    {
        return Widgets::create(...$widgets);
    }
    
    
    /**
     * Return widget definitions.
     */
    public function createWidgetDefinitions(WidgetDefinition ...$widgetDefinitions): WidgetDefinitions
    {
        return WidgetDefinitions::create(...$widgetDefinitions);
    }
    
    
    /**
     * Return widget database entry.
     */
    public function createWidgetDatabaseEntry(
        Type $type,
        Label $label,
        Width $width,
        Height $height,
        HorizontalPosition $horizontalPosition,
        VerticalPosition $verticalPosition,
        UpdateSet $optionUpdateSet,
        ?Id $id = null
    ): WidgetDatabaseEntry {
        return WidgetDatabaseEntry::create($type,
                                           $label,
                                           $width,
                                           $height,
                                           $horizontalPosition,
                                           $verticalPosition,
                                           $optionUpdateSet,
                                           $id);
    }
    
    
    /**
     * Return widget database entries.
     */
    public function createWidgetDatabaseEntries(WidgetDatabaseEntry ...$databaseEntries): WidgetDatabaseEntries
    {
        return WidgetDatabaseEntries::create(...$databaseEntries);
    }
    
    
    /**
     * Return ID.
     */
    public function createId(int $id): Id
    {
        return Id::create($id);
    }
    
    
    /**
     * Return label.
     */
    public function createLabel(string $title): Label
    {
        return Label::create($title);
    }
    
    
    /**
     * Return height.
     */
    public function createHeight(int $height): Height
    {
        return Height::create($height);
    }
    
    
    /**
     * Return width.
     */
    public function createWidth(int $width): Width
    {
        return Width::create($width);
    }
    
    
    /**
     * Return horizontal position.
     */
    public function createHorizontalPosition(int $horizontalPosition): HorizontalPosition
    {
        return HorizontalPosition::create($horizontalPosition);
    }
    
    
    /**
     * Return vertical position.
     */
    public function createVerticalPosition(int $verticalPosition): VerticalPosition
    {
        return VerticalPosition::create($verticalPosition);
    }
    
    
    /**
     * Return title.
     */
    public function createTitle(LanguageCode $languageCode, string $title): Title
    {
        return Title::create($languageCode, $title);
    }
    
    
    /**
     * Return titles.
     */
    public function createTitles(Title ...$titles): Titles
    {
        return Titles::create(...$titles);
    }
    
    
    /**
     * Return version.
     */
    public function createVersion(string $version): Version
    {
        return Version::create($version);
    }
    
    
    /**
     * Return icon.
     */
    public function createIcon(string $icon): Icon
    {
        return Icon::create($icon);
    }
    
    
    /**
     * Return icon using data.
     */
    public function createIconUsingData(string $data): Icon
    {
        return Icon::createUsingData($data);
    }
    
    
    /**
     * Return author.
     */
    public function createAuthor(string $author): Author
    {
        return Author::create($author);
    }
    
    
    /**
     * Return type.
     */
    public function createType(string $type): Type
    {
        return Type::create($type);
    }
    
    
    /**
     * Return language code.
     */
    public function createLanguageCode(string $languageCode): LanguageCode
    {
        return $this->languageFactory->createLanguageCode($languageCode);
    }
}